#include <hiveframework.h>
#include <inttypes.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#ifdef _WIN32
#  include <Windows.h>
#else
#  include <unistd.h>
#endif


static void log_record(HIVE_OUTCOLUMN clm, hive_time_t ts, const HIVE_INCOLUMN *colinfo, const HIVE_RECORD_DATA *rd);


/********************************************************************/
/* Collector Implementation Example                                */
/********************************************************************/
static bool premain(HIVE_COMPONENT comp, const char *param, HIVE_STATUS status) {
  HIVE_LOG_INFO("premain called");
  return true;
}

static void postmain(HIVE_COMPONENT comp, const char *param) {
  HIVE_LOG_INFO("postmain called");
}

static bool mainloop(HIVE_COMPONENT comp, const char *param, HIVE_STATUS status) {
  HIVE_LOG_INFO("cont_read mainloop begin.");
  HIVE_OUTCOLUMN clm_log = hive_outport_create_column(comp, OUTPORT1, "log", HIVE_DATA_SCALAR(HIVE_TYPE_STRING), HIVE_COLUMN_OPTION_NONE);

  HIVE_CONTINUOUS_READER *creader
    = hive_inport_continuous_reader(comp, INPORT1, hive_timestamp()+5000000000UL);

  hive_time_t last_updated = 0;
  while (hive_component_runnable(comp)) {
    const HIVE_COLUMN_READ_RESULT *read_result = hive_continuous_reader_read(creader);
    if (!read_result) {
      // NULL is returned when there is no valid window available in the range
      HIVE_LOG_INFO("no window");
      continue;
    }

    // Output column names to log when column information changes
    if (last_updated < read_result->incolumn_list->updated_at) {
      last_updated = read_result->incolumn_list->updated_at;
      HIVE_LOG_INFO("updated_at: %" PRIu64, last_updated);
      // Header section for log output
      //log_column_names(clm_log, read_result);
    }

    // Iteration of data & aggregation records
    HIVE_RECORD_ITERATOR iter = HIVE_GET_RECORD_ITERATOR(read_result);
    const HIVE_RECORD *record;
    while ((record = HIVE_RECORD_ITERATOR_GET_NEXT(&iter)) != NULL) {
      // Log output of data & aggregation records
      for (int i=0; i < record->data_count; i++) {
        const HIVE_RECORD_DATA *rd = HIVE_RECORD_GET_DATA(record, i);
        if (rd && rd->data_size != 0) {
          log_record(clm_log, record->timestamp, &read_result->incolumn_list->incolumns[i], rd);
        }
      }
    }
  }
  hive_continuous_reader_release(creader);

  HIVE_LOG_INFO("cont_read mainloop end.");
  return true;
}

static void log_record(HIVE_OUTCOLUMN clm, hive_time_t ts, const HIVE_INCOLUMN *colinfo, const HIVE_RECORD_DATA *rd) {
  const char *sourcename   = colinfo->source_component;
  const char *dataname     = colinfo->data_name;
  //HIVE_DATA_TYPE type      = colinfo->data_type;
  const char *dptr         = rd->data;
  //size_t datasize          = rd->data_size;

  // Generate output string
  char buff[128];
  switch (colinfo->data_type.type) {
  case HIVE_TYPE_BOOLEAN:
    snprintf(buff, 128, "%s:%s => %s", sourcename, dataname, (*(const int8_t*)dptr)?"true":"false");
    break;
  case HIVE_TYPE_INT8:
    snprintf(buff, 128, "%s:%s => %d", sourcename, dataname, (int)*(const int8_t*)dptr);
    break;
  case HIVE_TYPE_INT16:
    snprintf(buff, 128, "%s:%s => %d", sourcename, dataname, (int)*(const int16_t*)dptr);
    break;
  case HIVE_TYPE_INT32:
    snprintf(buff, 128, "%s:%s => %d", sourcename, dataname, (int)*(const int32_t*)dptr);
    break;
  case HIVE_TYPE_INT64:
    snprintf(buff, 128, "%s:%s => %" PRIi64, sourcename, dataname, *(const int64_t*)dptr);
    break;
  case HIVE_TYPE_UINT8:
    snprintf(buff, 128, "%s:%s => %u", sourcename, dataname, (unsigned int)*(const uint8_t*)dptr);
    break;
  case HIVE_TYPE_UINT16:
    snprintf(buff, 128, "%s:%s => %u", sourcename, dataname, (unsigned int)*(const uint16_t*)dptr);
    break;
  case HIVE_TYPE_UINT32:
    snprintf(buff, 128, "%s:%s => %u", sourcename, dataname, (unsigned int)*(const uint32_t*)dptr);
    break;
  case HIVE_TYPE_UINT64:
    snprintf(buff, 128, "%s:%s => %" PRIu64, sourcename, dataname, *(const uint64_t*)dptr);
    break;
  case HIVE_TYPE_FLOAT:
    snprintf(buff, 128, "%s:%s => %f", sourcename, dataname, *(const float*)dptr);
    break;
  case HIVE_TYPE_DOUBLE:
    snprintf(buff, 128, "%s:%s => %lf", sourcename, dataname, *(const double*)dptr);
    break;
  case HIVE_TYPE_TIMESTAMP:
    snprintf(buff, 128, "%s:%s => %" PRIu64, sourcename, dataname, *(const uint64_t*)dptr);
    break;
  case HIVE_TYPE_STRING:
    snprintf(buff, 128, "%s:%s => %s", sourcename, dataname, dptr);
    break;
  default:
    snprintf(buff, 128, "%s:%s => (unsupport data type)", sourcename, dataname);
    break;
  }

  // Log output
  HIVE_LOG_INFO("DATA: %s", buff);

  // Output to string type column
  if (!hive_outcolumn_insert(clm, buff)) {
    HIVE_API_ERROR err = hive_get_api_error();
    HIVE_LOG_ERROR("insert 'count' failed: errcode=0x%08x", err.code);
  }
}

/* Collector definition data: describes name, description, parameters, and output data. */
const static HIVE_COMPONENT_DEFINITION info = {
  .uuid            = "590baab1-bdfe-4098-8917-cf4a8fa633bb",
  .name            = "read and log",
  .parameter_type  = HIVE_COMPONENT_PARAMETER_NONE,
  .in_ports        = 1,
  .out_ports       = 1,
  .functions     = {
    .premain            = premain,      // optional
    .main               = mainloop,     // required
    .postmain           = postmain,     // optional
  },
};
EXPORT_COMPONENT_DEFINITION(info);

